sh = function () {
    return "try sh.help();";
};

sh._checkMongos = function () {
    var result = db.isMaster();
    if (result.msg != "isdbgrid")
        nav_throwError("not connected to a mongos");
};

sh._checkFullName = function (fullName) {
    if (!fullName)
        nav_throwError("need a full name")
    if (fullName.indexOf(".") <= 0)
        nav_throwError("name needs to be <db>.<collection>'")
};

sh.getBalancerLockDetails = function (configDB) {    
    if (isUndefined(configDB))
        configDB = db.getSiblingDB('config');
    var lock = configDB.locks.findOne({ _id: 'balancer' });
    if (lock == null)
        return null;
    if (lock.state == 0)
        return null;
    return lock;
}

sh.abortReshardCollection = function (namespace) {
    if (!isString(namespace))
        nav_throwError("'namespace' parameter is " + typeof namespace + " but not string");

    this._checkFullName(namespace);

    return db.adminCommand({ abortReshardCollection: namespace });
};

sh.addShard = function (url) {
    return db.adminCommand({ addShard: url });
};

sh.addShardTag = function (shard, tag) {
    var version = db.getDatabaseVersion();
    if (version >= 30400) {
        return sh.addShardToZone(shard, tag);
    } else {
        this._checkMongos();
        var configDB = db.getSiblingDB("config");
        if (configDB.shards.findOne({ _id: shard }) == null)
            throw Error("can't find a shard with name: " + shard);
        return configDB.shards.update(
            { _id: shard }, { $addToSet: { tags: tag } }, { writeConcern: { w: 'majority', wtimeout: 60000 } });
    }
};

sh.addShardToZone = function (shardName, zoneName) {
    this._checkMongos();
    return db.adminCommand({ addShardToZone: shardName, zone: zoneName });
};

sh.addTagRange = function (namespace, minimum, maximum, tag) {
    var version = db.getDatabaseVersion();
    if (version >= 30400) {
        return sh.updateZoneKeyRange(namespace, minimum, maximum, tag);
    } else {
        if (JSON.stringify(minimum) == JSON.stringify(maximum))
            nav_throwError("minimum and maximum cannot be the same");

        this._checkMongos();
        var configDB = db.getSiblingDB("config");
        return configDB.tags.update({ _id: { ns: namespace, min: minimum } },
                                    { _id: { ns: namespace, min: minimum }, ns: namespace, min: minimum, max: maximum, tag: tag },
                                    { upsert: true, writeConcern: { w: 'majority', wtimeout: 60000 } });
    }
};

sh.balancerCollectionStatus = function (namespace) {
    if (!isString(namespace))
        nav_throwError("'namespace' parameter is " + typeof namespace + " but not string");
    this._checkFullName(namespace);

    return db.adminCommand({ balancerCollectionStatus: namespace });
};

sh.checkMetadataConsistency = function (options = {}) {
    if (!isObject(options))
        nav_throwError("'options' parameter is " + typeof options + " but not object");

    var result = db.adminCommand(Object.extend({ checkMetadataConsistency: 1 }, options));
    if (!result.ok)
        nav_throwError(result.errmsg, result);

    return result.cursor.firstBatch;
};

sh.commitReshardCollection = function (namespace) {
    if (!isString(namespace))
        nav_throwError("'namespace' parameter is " + typeof namespace + " but not string");
    this._checkFullName(namespace);

    return db.adminCommand({ commitReshardCollection: namespace });
};

sh.disableAutoMerger = function (namespace) {
    if (!isString(namespace))
        nav_throwError("'namespace' parameter is " + typeof namespace + " but not string");
    this._checkFullName(namespace);
        
    return db.getSiblingDB("config").getCollection('collections').updateOne(
            { _id: namespace },
            { $set: { enableAutoMerge: false } },
            { writeConcern: { w: 'majority', wtimeout: 60000 } });
};

sh.disableAutoSplit = function (configDB) {
    if (configDB === undefined) {
        this._checkMongos();
        configDB = db.getSiblingDB("config");
    }
    return configDB.settings.update({ _id: 'autosplit' },
                                    { $set: { enabled: false } },
                                    { upsert: true, writeConcern: { w: 'majority', wtimeout: 30000 } });
};

sh.disableBalancing = function (namespace) {
    if (isUndefined(namespace))
        nav_throwError("Must specify collection");

    sh._checkMongos();
    return db.getSiblingDB("config").collections.update(
        { _id: namespace },
        { $set: { "noBalance": true } },
        { writeConcern: { w: 'majority', wtimeout: 60000 } });
};

sh.enableAutoMerger = function (namespace) {
    if (!isString(namespace))
        nav_throwError("'namespace' parameter is " + typeof namespace + " but not string");
    this._checkFullName(namespace);

    return db.getSiblingDB("config").getCollection('collections').updateOne(
        { _id: namespace },
        { $set: { enableAutoMerge: 1 } },
        { writeConcern: { w: 'majority', wtimeout: 60000 } });
};

sh.enableAutoSplit = function (configDB) {
    if (configDB === undefined) {
        this._checkMongos();
        configDB = db.getSiblingDB("config");
    }
    return configDB.settings.update({ _id: 'autosplit' },
                                    { $set: { enabled: true } },
                                    { upsert: true, writeConcern: { w: 'majority', wtimeout: 30000 } });
};

sh.enableBalancing = function (namespace) {
    if (isUndefined(namespace))
        nav_throwError("Must specify collection");

    sh._checkMongos();
    return db.getSiblingDB("config").collections.update(
        { _id: namespace + "" },
        { $set: { "noBalance": false } },
        { writeConcern: { w: 'majority', wtimeout: 60000 } });
};

sh.enableSharding = function (dbname, primaryShard) {
    if (!isString(dbname))
        nav_throwError("need a valid dbname");

    this._checkMongos();

    var command = { enableSharding: dbname };
    if (!isUndefined(primaryShard) && isString(primaryShard))
        command.primaryShard = primaryShard;

    return db.adminCommand(command);
};

sh.getActiveMigrations = function (configDB) {
    if (isUndefined(configDB)) {
        this._checkMongos();
        configDB = db.getSiblingDB("config");
    }
    var activeLocks = configDB.locks.find({ state: { $eq: 2 } });
    var result = [];
    if (activeLocks != null) {
        activeLocks.forEach(function (lock) {
            result.push({ _id: lock._id, when: lock.when });
        });
    }
    return result;
};

sh.getBalancerHost = function (configDB) {
    if (isUndefined(configDB)) {
        this._checkMongos();
        configDB = db.getSiblingDB("config");
    }
    var locks = configDB.locks.findOne({ _id: "balancer" });
    if (locks == null)
        return "config.locks collection does not contain balancer lock. be sure you are connected to a mongos";
    else if (locks.process.match(/ConfigServer/))
        return "getBalancerHost is deprecated starting version 3.4. The balancer is running on the config server primary host.";
    else
        return locks.process.match(/[^:]+:[^:]+/)[0];
}

sh.getBalancerState = function (configDB) {
    if (isUndefined(configDB)) {
        this._checkMongos();
        configDB = db.getSiblingDB("config");
    }
    var settings = configDB.settings.findOne({ _id: "balancer" });
    if (settings == null)
        return true;
    return !settings.stopped;
};

sh.getBalancerWindow = function (configDB) {
    if (isUndefined(configDB)) {
        this._checkMongos();
        configDB = db.getSiblingDB("config");
    }
    var settings = configDB.settings.findOne({ _id: 'balancer' });
    if (settings == null)
        return null;
    if (settings.hasOwnProperty("activeWindow"))
        return settings.activeWindow;
    return null;
};

sh.getRecentFailedRounds = function (configDB) {
    if (isUndefined(configDB))
        configDB = db.getSiblingDB("config");
    var balErrs = configDB.actionlog.find({ what: "balancer.round" }).sort({ time: -1 }).limit(5);
    var result = { count: 0, lastErr: "", lastTime: " " };
    if (balErrs != null) {
        balErrs.forEach(function (r) {
            if (r.details.errorOccured) {
                result.count += 1;
                result.lastErr = r.details.errmsg;
                result.lastTime = r.time;
            }
        });
    }
    return result;
};

sh.getRecentMigrations = function (configDB) {
    if (isUndefined(configDB))
        configDB = db.getSiblingDB("config");
    var yesterday = new Date(new Date() - 24 * 60 * 60 * 1000);

    var result = configDB.changelog
                     .aggregate([
                         {
                             $match: {
                                 time: { $gt: yesterday },
                                 what: "moveChunk.from",
                                 'details.errmsg': { $exists: false },
                                 'details.note': 'success'
                             }
                         },
                         { $group: { _id: { msg: "$details.errmsg" }, count: { $sum: 1 } } },
                         { $project: { _id: { $ifNull: ["$_id.msg", "Success"] }, count: "$count" } }
                     ])
                     .toArray();

    result = result.concat(
        configDB.changelog
            .aggregate([
                {
                    $match: {
                        time: { $gt: yesterday },
                        what: "moveChunk.from",
                        $or: [
                            { 'details.errmsg': { $exists: true } },
                            { 'details.note': { $ne: 'success' } }
                        ]
                    }
                },
                {
                    $group: {
                        _id: { msg: "$details.errmsg", from: "$details.from", to: "$details.to" },
                        count: { $sum: 1 }
                    }
                },
                {
                    $project: {
                        _id: { $ifNull: ['$_id.msg', 'aborted'] },
                        from: "$_id.from",
                        to: "$_id.to",
                        count: "$count"
                    }
                }
            ])
            .toArray());

    return result;
};

sh.getShardedDataDistribution = function (options = {}) {
    var cursor = db.getSiblingDB('admin').aggregate([{ $shardedDataDistribution: options }]);
    cursor.hasNext();
    return cursor;
};

sh.getShouldAutoSplit = function (configDB) {
    if (isUndefined(configDB)) {
        this._checkMongos();
        configDB = db.getSiblingDB("config");
    }
    var autosplit = configDB.settings.findOne({ _id: 'autosplit' });
    if (autosplit == null)
        return true;
    return autosplit.enabled;
};

sh.help = function () {
    print(
        "\tsh.abortReshardCollection(fullName) - abort the current reshardCollection on a given collection\n" +
        "\tsh.addShard(host) - server:port OR setname/server:port\n" +
        "\tsh.addShardTag(shard,tag) - calls addShardTag for a sharded DB\n" +
        "\tsh.addShardToZone(shard,zone) - adds the shard to the zone\n" +
        "\tsh.balancerCollectionStatus(fullName) - returns information on whether the chunks of a sharded collection are balanced\n" +
        "\tsh.checkMetadataConsistency(options) - returns a cursor with information about metadata inconsistencies\n" +
        "\tsh.commitReshardCollection(fullName) - commits the current reshardCollection on a given collection\n" +
        "\tsh.disableAutoMerger(fullName) - disable auto-merging on one collection\n" +
        "\tsh.disableAutoSplit() - disable autoSplit on one collection\n" +
        "\tsh.disableBalancing(namespace) - disable balancing on one collection\n" +
        "\tsh.enableAutoMerger(namespace) - enable auto-merging on one collection\n" +
        "\tsh.enableAutoSplit() - re-enable autoSplit on one collection\n" +
        "\tsh.enableBalancing(coll) - re-enable balancing on one collection\n" +
        "\tsh.enableSharding(dbname) - enables sharding on the database dbname\n" +
        "\tsh.getBalancerState() - returns whether the balancer is enabled\n" +
        "\tsh.getShardedDataDistribution() - returns data-size distribution information for all existing sharded collections\n" +
        "\tsh.getShouldAutoSplit() - returns whether autosplit is enabled\n" +
        "\tsh.isBalancerRunning() - return true if the balancer has work in progress on any mongos\n" +
        "\tsh.moveChunk(fullName,find,to) - move the chunk where 'find' is to 'to' (name of shard)\n" +
        "\tsh.removeShardFromZone(shard,zone) - removes the shard from zone\n" +
        "\tsh.removeRangeFromZone(fullName,min,max) - removes the range of the given collection from any zone\n" +
        "\tsh.reshardCollection(namespace,key,unique,options) - enables sharding for a collection. Uses the reshardCollection command\n" +
        "\tsh.shardCollection(fullName,key,unique,options) - shards the collection\n" +
        "\tsh.splitAt(fullName,middle) - splits the chunk that middle is in at middle\n" +
        "\tsh.splitFind(fullName,find) - plits the chunk that find is in at the median\n" +
        "\tsh.startAutoMerger() - globally enable auto-merger (active only if balancer is up)\n" +
        "\tsh.startBalancer() - starts the balancer so chunks are balanced automatically\n" +
        "\tsh.status() - prints a general overview of the cluster\n" +
        "\tsh.stopAutoMerger() - globally disable auto-merger\n" +
        "\tsh.stopBalancer() - stops the balancer so chunks are not balanced automatically\n" +
        "\tsh.updateZoneKeyRange(fullName,min,max,zone) - assigns the specified range of the given collection to a zone\n");
};

sh.isBalancerRunning = function (configDB) {
    if (isUndefined(configDB)) {
        this._checkMongos();
        configDB = db.getSiblingDB("config");
    }
    
    var version = db.getDatabaseVersion();
    if (version >= 30400) {
        var result = configDB.adminCommand({ balancerStatus: 1 });
        if (!result.ok)
            nav_throwError(result.errmsg, result);
        return result.inBalancerRound;
    } else {
        var locks = configDB.locks.findOne({ _id: "balancer" });
        if (locks == null) {
            print("config.locks collection empty or missing. be sure you are connected to a mongos");
            return false;
        }
        return locks.state > 0;
    }
};

sh.moveChunk = function (namespace, query, destination) {
    this._checkFullName(namespace);
    this._checkMongos();
    return db.adminCommand({ moveChunk: namespace, find: query, to: destination });
};

sh.removeShardTag = function (shard, tag) {
    var version = db.getDatabaseVersion();
    if (version >= 30400) {
        return sh.removeShardFromZone(shard, tag);
    } else {
        this._checkMongos();
        var configDB = db.getSiblingDB("config");
        if (configDB.shards.findOne({ _id: shard }) == null)
            nav_throwError("can't find a shard with name: " + shard);

        return configDB.shards.update({ _id: shard }, { $pull: { tags: tag } }, { writeConcern: { w: 'majority', wtimeout: 60000 } });
    }
};

sh.removeShardFromZone = function (shardName, zoneName) {
    this._checkMongos();
    return db.adminCommand({ removeShardFromZone: shardName, zone: zoneName });
};

sh.removeTagRange = function (namespace, minimum, maximum, tag) {
    var version = db.getDatabaseVersion();
    if (version >= 30400) {
        return sh.removeRangeFromZone(namespace, minimum, maximum);
    } else {
        this._checkMongos();
        var configDB = db.getSiblingDB("config");
        if (configDB.collections.findOne({ _id: namespace }) == null) {
            print("Warning: can't find the namespace: " + namespace + " - collection likely never sharded");
        }
        if (configDB.shards.findOne({ tags: tag })) {
            print("Warning: tag still in use by at least one shard");
        }
        return configDB.tags.remove({ _id: { ns: namespace, min: minimum }, max: maximum, tag: tag },
                                    { writeConcern: { w: 'majority', wtimeout: 60000 } });
    }
};

sh.removeRangeFromZone = function (namespace, minimum, maximum) {
    this._checkMongos();
    return db.adminCommand({ updateZoneKeyRange: namespace, min: minimum, max: maximum, zone: null });
};

sh.reshardCollection = function (namespace, key, unique, options) {
    if (!isString(namespace))
        nav_throwError("'namespace' parameter is " + typeof namespace + " but not string");
    this._checkFullName(namespace);
    if (!isObject(key))
        nav_throwError("'key' parameter is " + typeof key + " but not object");
        
    if (isObject(unique) && unique != null) {
        options = unique;
        unique = undefined;
    }

    var commandObj = {
        reshardCollection: namespace,
        key: key
    };
    if (!isUndefined(unique)) {
        commandObj.unique = unique;
    }

    return db.adminCommand(Object.extend(commandObj, options));
};

sh.setBalancerState = function (isOn) {
    if (isOn)
        return sh.startBalancer();
    else 
        return sh.stopBalancer();
};

sh.shardCollection = function (namespace, key, unique, options) {
    this._checkFullName(namespace);
    if (isUndefined(key))
        nav_throwError("need a key")
    if (!isObject(key))
        nav_throwError("key needs to be an object");

    var commandObj = { shardCollection: namespace, key: key };
    if (unique)
        commandObj.unique = true;
    if (options) {
        if (typeof (options) !== "object") {
            throw new Error("options must be an object");
        }
        Object.extend(commandObj, options);
    }

    this._checkMongos();
    return db.adminCommand(commandObj);
};

sh.splitAt = function (namespace, query) {
    this._checkFullName(namespace);
    this._checkMongos();
    return db.adminCommand({ split: namespace, middle: query });
};

sh.splitFind = function (namespace, query) {
    this._checkFullName(namespace);
    this._checkMongos();
    return db.adminCommand({ split: namespace, find: query });
};

sh.startAutoMerger = function () {
    return db.getSiblingDB("config").getCollection('settings').updateOne(
        { _id: 'automerge' },
        { $set: { enabled: true } },
        { upsert: true, writeConcern: { w: 'majority', wtimeout: 30000 } });
};

sh.startBalancer = function (timeout, interval) {
    timeout = timeout || 60000;

    var version = db.getDatabaseVersion();
    if (version >= 30400) {
        return db.adminCommand({ balancerStart: 1, maxTimeMS: timeout });
    } else {
        var result = sh._writeBalancerStateDeprecated(false);
        sh.waitForBalancer(false, timeout, interval);
        return result;
    }
};

sh.status = function (verbose) {
    return db.printShardingStatus(verbose);
};

sh.stopAutoMerger = function () {
    return db.getSiblingDB("config").getCollection('settings').updateOne(
        { _id: 'automerge' },
        { $set: { enabled: false } },
        { upsert: true, writeConcern: { w: 'majority', wtimeout: 30000 } });
};

sh.stopBalancer = function (timeout, interval) {
    timeout = timeout || 60000;

    var version = db.getDatabaseVersion();
    if (version >= 30400) {
        return db.adminCommand({ balancerStop: 1, maxTimeMS: timeout });
    } else {
        var result = sh._writeBalancerStateDeprecated(false);
        sh.waitForBalancer(false, timeout, interval);
        return result;
    }
};

sh.updateZoneKeyRange = function (namespace, minimum, maximum, zone) {
    this._checkMongos();
    return db.adminCommand({ updateZoneKeyRange: namespace, min: minimum, max: maximum, zone: zone });
};

sh._waitForDLock = function (lockId, onOrNot, timeout, interval) {
    var state = onOrNot;
    var configDB = sh._getConfigDB();

    var beginTS = undefined;
    if (state == undefined) {
        var currLock = configDB.locks.findOne({ _id: lockId });
        if (currLock != null)
            beginTS = currLock.ts;
    }

    var lockStateOk = function () {
        var lock = configDB.locks.findOne({ _id: lockId });

        if (state == false)
            return !lock || lock.state == 0;
        if (state == true)
            return lock && lock.state == 2;
        if (state == undefined)
            return (beginTS == undefined && lock) ||
                (beginTS != undefined && (!lock || lock.ts + "" != beginTS + ""));
        else
            return lock && lock.state == state;
    };

    assert.soon(
        lockStateOk,
        "Waited too long for lock " + lockId + " to " +
            (state == true ? "lock" : (state == false ? "unlock" : "change to state " + state)),
        timeout,
        interval);
};

sh.waitForBalancer = function (wait, timeout, interval) {
    if (wait) {
        sh._waitForDLock("balancer", undefined, timeout, interval);
    } else {
        var activePings = [];
        this._checkMongos();
        db.getSiblingDB("config").mongos.find().forEach(function (ping) {
            if (!ping.waiting)
                activePings.push(ping);
        });

        print("Waiting for active hosts...");
        activePings = sh.waitForPingChange(activePings, 60 * 1000);

        print("Waiting for the balancer lock...");

        try {
            sh._waitForDLock("balancer", false, 15 * 60 * 1000);
        } catch (e) {
            print(
                "Balancer still may be active, you must manually verify this is not the case using the config.changelog collection.");
            nav_throwError(e);
        }

        print("Waiting again for active hosts after balancer is off...");

        activePings = sh.waitForPingChange(activePings, 5 * 1000);

        activePings.forEach(function (activePing) {
            print("Warning : host " + activePing._id + " seems to have been offline since " +
                    activePing.ping);
        });
    }
}

sh.waitForPingChange = function (activePings, timeout, interval) {
    this._checkMongos();
    var configDB = db.getSiblingDB("config");
    var isPingChanged = function (activePing) {
        var newPing = configDB.mongos.findOne({ _id: activePing._id });
        return !newPing || newPing.ping + "" != activePing.ping + "";
    };

    var start = new Date();

    var remainingPings = [];
    for (var i = 0; i < activePings.length; i++) {
        var activePing = activePings[i];
        print("Waiting for active host " + activePing._id +
              " to recognize new settings... (ping : " + activePing.ping + ")");

        var timeout = timeout || 30000;
        var interval = interval || 200;
        while (isPingChanged(activePing) != true) {
            if ((new Date()).getTime() - start.getTime() > timeout) {
                print("Waited for active ping to change for host " + activePing._id +
                      ", a migration may be in progress or the host may be down.");
                remainingPings.push(activePing);
                break;
            }
            sleep(interval);
        }
    }

    return remainingPings;
};
